<?php
/**
 * Submit Field Value
 *
 * @package     AutomatorWP\Integrations\EverestForms\Triggers\Submit_Field_Value
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_Everest_Forms_Submit_Field_Value extends AutomatorWP_Integration_Trigger {

    public $integration = 'everest_forms';
    public $trigger = 'everest_forms_submit_field_value';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User submits a specific field value on a form', 'automatorwp-pro' ),
            'select_option'     => __( 'User submits <strong>a specific field value</strong> on a form', 'automatorwp-pro' ),
            /* translators: %1$s: Field name. %2$s: Field value. %3$s: Post title. %4$s: Number of times. */
            'edit_label'        => sprintf( __( 'User submits the %1$s field with the value %2$s on %3$s %4$s time(s)', 'automatorwp-pro' ), '{field_name}', '{field_value}', '{post}', '{times}' ),
            /* translators: %1$s: Field name. %2$s: Field value. %3$s: Post title. */
            'log_label'         => sprintf( __( 'User submits the %1$s field with the value %2$s on %3$s', 'automatorwp-pro' ), '{field_name}', '{field_value}', '{post}' ),
            'action'            => 'everest_forms_process_complete',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 4,
            'options'           => array(
                'field_name' => array(
                    'from' => 'field_name',
                    'default' => __( 'field meta key', 'automatorwp-pro' ),
                    'fields' => array(
                        'field_name' => array(
                            'name' => __( 'Field meta key:', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => ''
                        )
                    )
                ),
                'field_value' => array(
                    'from' => 'field_value',
                    'default' => __( 'field value', 'automatorwp-pro' ),
                    'fields' => array(
                        'field_value' => array(
                            'name' => __( 'Field value:', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => ''
                        )
                    )
                ),
                'post' => automatorwp_utilities_post_option(array(
                    'name' => __( 'Form:', 'automatorwp-pro' ),
                    'option_none_label' => __('any form', 'automatorwp-pro'),
                    'post_type' => 'everest_form'
                )),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                array(
                    'form_field:FIELD_NAME' => array(
                        'label'     => __( 'Form field value', 'automatorwp-pro' ),
                        'type'      => 'text',
                        'preview'   => __( 'Form field value, replace "FIELD_NAME" by the field name', 'automatorwp-pro' ),
                    ),
                ),
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     * 
     * @param array $form_fields
     * @param array $entry
     * @param array $form_data
     * @param array $entry_id
     * @param \EverestForms\App\Modules\Form\Form $form
     */
    public function listener($form_fields, $entry, $form_data, $entry_id)
    {

        $form_id = $form_data['id'];
        $user_id = get_current_user_id();

        // Login is required
        if ( $user_id === 0 ) {
            return;
        }

        $form_fields = automatorwp_everest_forms_get_form_fields_values( $form_fields );

        // Loop all fields to trigger events per field value
        foreach ( $form_fields as $field_name => $field_value ) {

            // Trigger submit form field value event
            automatorwp_trigger_event( array(
                'trigger'       => $this->trigger,
                'user_id'       => $user_id,
                'form_id'       => $form_id,
                'field_name'    => $field_name,
                'field_value'   => $field_value,
                'form_fields'   => $form_fields,
            ) );

        }

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Don't deserve if post, field name and value are not received
        if( ! isset( $event['form_id'] ) && ! isset( $event['field_name'] ) && ! isset( $event['field_value'] ) ) {
            return false;
        }

        // Bail if post doesn't match with the trigger option
        if( $trigger_options['post'] !== 'any' && absint( $event['form_id'] ) !== absint( $trigger_options['post'] ) ) {
            return false;
        }

        // Don't deserve if field name doesn't match with the trigger option
        if( $event['field_name'] !== $trigger_options['field_name'] ) {
            return false;
        }

        // Check if field value matches the required one (with support for arrays)
        if( is_array( $event['field_value'] ) ) {

            if( ! in_array( $trigger_options['field_value'], $event['field_value'] ) ) {
                return false;
            }

        } else {

            if( $event['field_value'] !== $trigger_options['field_value'] ) {
                return false;
            }

        }

        return $deserves_trigger;

    }

    /**
     * Register the required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_trigger_log_meta', array( $this, 'log_meta' ), 10, 6 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();
    }

    /**
     * Trigger custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return array
     */
    function log_meta( $log_meta, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Bail if action type don't match this action
        if( $trigger->type !== $this->trigger ) {
            return $log_meta;
        }

        $log_meta['form_fields'] = ( isset( $event['form_fields'] ) ? $event['form_fields'] : array() );

        return $log_meta;

    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an trigger
        if( $log->type !== 'trigger' ) {
            return $log_fields;
        }

        // Bail if trigger type don't match this trigger
        if( $object->type !== $this->trigger ) {
            return $log_fields;
        }

        $log_fields['form_fields'] = array(
            'name' => __( 'Fields Submitted', 'automatorwp-pro' ),
            'desc' => __( 'Information about the fields values sent on this form submission.', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;

    }

}

new AutomatorWP_Everest_Forms_Submit_Field_Value();